/*
 * Copyright 2016-2018 Józef Kucia for CodeWeavers
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#ifndef __VKD3D_D3D12_TEST_UTILS_H
#define __VKD3D_D3D12_TEST_UTILS_H

#include "utils.h"

struct d3d12_root_signature_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    ID3D12RootSignature *root_signature;
};

struct d3d12_shader_bytecode_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_SHADER_BYTECODE shader_bytecode;
};

struct d3d12_stream_output_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_STREAM_OUTPUT_DESC stream_output_desc;
};

struct d3d12_blend_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_BLEND_DESC blend_desc;
};

struct d3d12_sample_mask_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    UINT sample_mask;
};

struct d3d12_rasterizer_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_RASTERIZER_DESC rasterizer_desc;
};

struct d3d12_depth_stencil_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_DEPTH_STENCIL_DESC depth_stencil_desc;
};

struct d3d12_input_layout_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_INPUT_LAYOUT_DESC input_layout;
};

struct d3d12_ib_strip_cut_value_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_INDEX_BUFFER_STRIP_CUT_VALUE strip_cut_value;
};

struct d3d12_primitive_topology_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_PRIMITIVE_TOPOLOGY_TYPE primitive_topology_type;
};

struct d3d12_render_target_formats_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    struct D3D12_RT_FORMAT_ARRAY render_target_formats;
};

struct d3d12_depth_stencil_format_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    DXGI_FORMAT depth_stencil_format;
};

struct d3d12_sample_desc_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    DXGI_SAMPLE_DESC sample_desc;
};

struct d3d12_node_mask_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    UINT node_mask;
};

struct d3d12_cached_pso_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_CACHED_PIPELINE_STATE cached_pso;
};

struct d3d12_flags_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_PIPELINE_STATE_FLAGS flags;
};

struct d3d12_depth_stencil1_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_DEPTH_STENCIL_DESC1 depth_stencil_desc;
};

struct d3d12_view_instancing_subobject
{
    DECLSPEC_ALIGN(sizeof(void *)) D3D12_PIPELINE_STATE_SUBOBJECT_TYPE type;
    D3D12_VIEW_INSTANCING_DESC view_instancing_desc;
};

#define wait_queue_idle(a, b) wait_queue_idle_(__FILE__, __LINE__, a, b)
static void wait_queue_idle_(const char *file, unsigned int line, ID3D12Device *device, ID3D12CommandQueue *queue);
static ID3D12Device *create_device(void);

static inline HRESULT wait_for_fence_no_event(ID3D12Fence *fence, uint64_t value)
{
    if (ID3D12Fence_GetCompletedValue(fence) >= value)
        return S_OK;

    /* This is defined to block on the value with infinite timeout. */
    return ID3D12Fence_SetEventOnCompletion(fence, value, NULL);
}

#define wait_queue_idle_no_event(a, b) wait_queue_idle_no_event_(__FILE__, __LINE__, a, b)
static inline void wait_queue_idle_no_event_(const char *file,
        unsigned int line, ID3D12Device *device, ID3D12CommandQueue *queue)
{
    ID3D12Fence *fence;
    HRESULT hr;

    hr = ID3D12Device_CreateFence(device, 0, D3D12_FENCE_FLAG_NONE,
            &IID_ID3D12Fence, (void **)&fence);
    assert_that_(file, line)(hr == S_OK, "Failed to create fence, hr %#x.\n", hr);

    hr = ID3D12CommandQueue_Signal(queue, fence, 1);
    assert_that_(file, line)(hr == S_OK, "Failed to signal fence, hr %#x.\n", hr);
    hr = wait_for_fence_no_event(fence, 1);
    assert_that_(file, line)(hr == S_OK, "Failed to wait for fence, hr %#x.\n", hr);

    ID3D12Fence_Release(fence);
}

static inline void set_box(D3D12_BOX *box, unsigned int left, unsigned int top, unsigned int front,
        unsigned int right, unsigned int bottom, unsigned int back)
{
    box->left = left;
    box->top = top;
    box->front = front;
    box->right = right;
    box->bottom = bottom;
    box->back = back;
}

static void set_viewport(D3D12_VIEWPORT *vp, float x, float y,
        float width, float height, float min_depth, float max_depth)
{
    vp->TopLeftX = x;
    vp->TopLeftY = y;
    vp->Width = width;
    vp->Height = height;
    vp->MinDepth = min_depth;
    vp->MaxDepth = max_depth;
}

static D3D12_SHADER_BYTECODE shader_bytecode(const DWORD *code, size_t size)
{
    D3D12_SHADER_BYTECODE shader_bytecode = { code, size };
    return shader_bytecode;
}

static inline D3D12_SHADER_BYTECODE shader_bytecode_from_blob(ID3D10Blob *blob)
{
    return shader_bytecode(ID3D10Blob_GetBufferPointer(blob), ID3D10Blob_GetBufferSize(blob));
}

static void exec_command_list(ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *list)
{
    ID3D12CommandList *lists[] = {(ID3D12CommandList *)list};
    ID3D12CommandQueue_ExecuteCommandLists(queue, 1, lists);
}

#define reset_command_list(a, b) reset_command_list_(__FILE__, __LINE__, a, b)
static inline void reset_command_list_(const char *file, unsigned int line,
        ID3D12GraphicsCommandList *list, ID3D12CommandAllocator *allocator)
{
    HRESULT hr;

    hr = ID3D12CommandAllocator_Reset(allocator);
    assert_that_(file, line)(hr == S_OK, "Failed to reset command allocator, hr %#x.\n", hr);
    hr = ID3D12GraphicsCommandList_Reset(list, allocator, NULL);
    assert_that_(file, line)(hr == S_OK, "Failed to reset command list, hr %#x.\n", hr);
}

#define queue_signal(a, b, c) queue_signal_(__FILE__, __LINE__, a, b, c)
static inline void queue_signal_(const char *file, unsigned int line,
        ID3D12CommandQueue *queue, ID3D12Fence *fence, uint64_t value)
{
    HRESULT hr;

    hr = ID3D12CommandQueue_Signal(queue, fence, value);
    ok_(file, line)(hr == S_OK, "Failed to submit signal operation to queue, hr %#x.\n", hr);
}

#define queue_wait(a, b, c) queue_wait_(__FILE__, __LINE__, a, b, c)
static inline void queue_wait_(const char *file, unsigned int line,
        ID3D12CommandQueue *queue, ID3D12Fence *fence, uint64_t value)
{
    HRESULT hr;

    hr = ID3D12CommandQueue_Wait(queue, fence, value);
    ok_(file, line)(hr == S_OK, "Failed to submit wait operation to queue, hr %#x.\n", hr);
}

#define create_placed_buffer(a, b, c, d, e, f) create_placed_buffer_(__FILE__, __LINE__, a, b, c, d, e, f)
static inline ID3D12Resource *create_placed_buffer_(const char *file, unsigned int line, ID3D12Device *device,
        ID3D12Heap *heap, size_t offset, size_t size, D3D12_RESOURCE_FLAGS resource_flags,
        D3D12_RESOURCE_STATES initial_resource_state)
{
    D3D12_RESOURCE_DESC resource_desc;
    ID3D12Resource *buffer;
    HRESULT hr;

    resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_BUFFER;
    resource_desc.Alignment = 0;
    resource_desc.Width = size;
    resource_desc.Height = 1;
    resource_desc.DepthOrArraySize = 1;
    resource_desc.MipLevels = 1;
    resource_desc.Format = DXGI_FORMAT_UNKNOWN;
    resource_desc.SampleDesc.Count = 1;
    resource_desc.SampleDesc.Quality = 0;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_ROW_MAJOR;
    resource_desc.Flags = resource_flags;

    hr = ID3D12Device_CreatePlacedResource(device, heap, offset, &resource_desc,
            initial_resource_state, NULL, &IID_ID3D12Resource, (void **)&buffer);
    assert_that_(file, line)(SUCCEEDED(hr), "Failed to create buffer, hr %#x.\n", hr);
    return buffer;
}

#define create_buffer(a, b, c, d, e) create_buffer_(__FILE__, __LINE__, a, b, c, d, e)
static ID3D12Resource *create_buffer_(const char *file, unsigned int line, ID3D12Device *device,
        D3D12_HEAP_TYPE heap_type, size_t size, D3D12_RESOURCE_FLAGS resource_flags,
        D3D12_RESOURCE_STATES initial_resource_state)
{
    D3D12_HEAP_PROPERTIES heap_properties;
    D3D12_RESOURCE_DESC resource_desc;
    ID3D12Resource *buffer;
    HRESULT hr;

    memset(&heap_properties, 0, sizeof(heap_properties));
    heap_properties.Type = heap_type;

    resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_BUFFER;
    resource_desc.Alignment = 0;
    resource_desc.Width = size;
    resource_desc.Height = 1;
    resource_desc.DepthOrArraySize = 1;
    resource_desc.MipLevels = 1;
    resource_desc.Format = DXGI_FORMAT_UNKNOWN;
    resource_desc.SampleDesc.Count = 1;
    resource_desc.SampleDesc.Quality = 0;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_ROW_MAJOR;
    resource_desc.Flags = resource_flags;

    hr = ID3D12Device_CreateCommittedResource(device, &heap_properties,
            D3D12_HEAP_FLAG_NONE, &resource_desc, initial_resource_state,
            NULL, &IID_ID3D12Resource, (void **)&buffer);
    assert_that_(file, line)(SUCCEEDED(hr), "Failed to create buffer, hr %#x.\n", hr);
    return buffer;
}

#define create_readback_buffer(a, b) create_readback_buffer_(__FILE__, __LINE__, a, b)
static ID3D12Resource *create_readback_buffer_(const char *file, unsigned int line, ID3D12Device *device,
        size_t size)
{
    return create_buffer_(file, line, device, D3D12_HEAP_TYPE_READBACK, size,
            D3D12_RESOURCE_FLAG_DENY_SHADER_RESOURCE, D3D12_RESOURCE_STATE_COPY_DEST);
}

#define update_buffer_data(a, b, c, d) update_buffer_data_(__FILE__, __LINE__, a, b, c, d)
static inline void update_buffer_data_(const char *file, unsigned int line, ID3D12Resource *buffer,
        size_t offset, size_t size, const void *data)
{
    D3D12_RANGE range;
    HRESULT hr;
    void *ptr;

    range.Begin = range.End = 0;
    hr = ID3D12Resource_Map(buffer, 0, &range, &ptr);
    ok_(file, line)(hr == S_OK, "Failed to map buffer, hr %#x.\n", hr);
    memcpy((BYTE *)ptr + offset, data, size);
    ID3D12Resource_Unmap(buffer, 0, NULL);
}

#define create_upload_buffer(a, b, c) create_upload_buffer_(__FILE__, __LINE__, a, b, c)
static inline ID3D12Resource *create_upload_buffer_(const char *file, unsigned int line, ID3D12Device *device,
        size_t size, const void *data)
{
    ID3D12Resource *buffer;

    buffer = create_buffer_(file, line, device, D3D12_HEAP_TYPE_UPLOAD, size,
            D3D12_RESOURCE_FLAG_NONE, D3D12_RESOURCE_STATE_GENERIC_READ);
    if (data)
        update_buffer_data_(file, line, buffer, 0, size, data);
    return buffer;
}

#define create_cpu_descriptor_heap(a, b, c) create_cpu_descriptor_heap_(__FILE__, __LINE__, a, b, c)
static inline ID3D12DescriptorHeap *create_cpu_descriptor_heap_(const char *file, unsigned int line,
        ID3D12Device *device, D3D12_DESCRIPTOR_HEAP_TYPE heap_type, unsigned int descriptor_count)
{
    D3D12_DESCRIPTOR_HEAP_DESC heap_desc;
    ID3D12DescriptorHeap *descriptor_heap;
    HRESULT hr;

    heap_desc.Type = heap_type,
    heap_desc.NumDescriptors = descriptor_count;
    heap_desc.Flags = D3D12_DESCRIPTOR_HEAP_FLAG_NONE;
    heap_desc.NodeMask = 0;
    hr = ID3D12Device_CreateDescriptorHeap(device, &heap_desc,
            &IID_ID3D12DescriptorHeap, (void **)&descriptor_heap);
    ok_(file, line)(SUCCEEDED(hr), "Failed to create descriptor heap, hr %#x.\n", hr);

    return descriptor_heap;
}

#define create_gpu_descriptor_heap(a, b, c) create_gpu_descriptor_heap_(__FILE__, __LINE__, a, b, c)
static inline ID3D12DescriptorHeap *create_gpu_descriptor_heap_(const char *file, unsigned int line,
        ID3D12Device *device, D3D12_DESCRIPTOR_HEAP_TYPE heap_type, unsigned int descriptor_count)
{
    D3D12_DESCRIPTOR_HEAP_DESC heap_desc;
    ID3D12DescriptorHeap *descriptor_heap;
    HRESULT hr;

    heap_desc.Type = heap_type,
    heap_desc.NumDescriptors = descriptor_count;
    heap_desc.Flags = D3D12_DESCRIPTOR_HEAP_FLAG_SHADER_VISIBLE;
    heap_desc.NodeMask = 0;
    hr = ID3D12Device_CreateDescriptorHeap(device, &heap_desc,
            &IID_ID3D12DescriptorHeap, (void **)&descriptor_heap);
    ok_(file, line)(SUCCEEDED(hr), "Failed to create descriptor heap, hr %#x.\n", hr);

    return descriptor_heap;
}

static void transition_sub_resource_state(ID3D12GraphicsCommandList *list, ID3D12Resource *resource,
        unsigned int sub_resource_idx, D3D12_RESOURCE_STATES state_before, D3D12_RESOURCE_STATES state_after)
{
    D3D12_RESOURCE_BARRIER barrier;

    barrier.Type = D3D12_RESOURCE_BARRIER_TYPE_TRANSITION;
    barrier.Flags = D3D12_RESOURCE_BARRIER_FLAG_NONE;
    barrier.Transition.pResource = resource;
    barrier.Transition.Subresource = sub_resource_idx;
    barrier.Transition.StateBefore = state_before;
    barrier.Transition.StateAfter = state_after;

    ID3D12GraphicsCommandList_ResourceBarrier(list, 1, &barrier);
}

static inline void uav_barrier(ID3D12GraphicsCommandList *list, ID3D12Resource *resource)
{
    D3D12_RESOURCE_BARRIER barrier;

    barrier.Type = D3D12_RESOURCE_BARRIER_TYPE_UAV;
    barrier.Flags = D3D12_RESOURCE_BARRIER_FLAG_NONE;
    barrier.UAV.pResource = resource;

    ID3D12GraphicsCommandList_ResourceBarrier(list, 1, &barrier);
}

#define create_command_queue(a, b, c) create_command_queue_(__FILE__, __LINE__, a, b, c)
static inline ID3D12CommandQueue *create_command_queue_(const char *file, unsigned int line, ID3D12Device *device,
        D3D12_COMMAND_LIST_TYPE type, int priority)
{
    D3D12_COMMAND_QUEUE_DESC queue_desc;
    ID3D12CommandQueue *queue;
    HRESULT hr;

    queue_desc.Type = type;
    queue_desc.Priority = priority;
    queue_desc.Flags = D3D12_COMMAND_QUEUE_FLAG_NONE;
    queue_desc.NodeMask = 0;
    hr = ID3D12Device_CreateCommandQueue(device, &queue_desc, &IID_ID3D12CommandQueue, (void **)&queue);
    ok_(file, line)(hr == S_OK, "Failed to create command queue, hr %#x.\n", hr);

    return queue;
}

static void transition_resource_state(ID3D12GraphicsCommandList *list, ID3D12Resource *resource,
        D3D12_RESOURCE_STATES state_before, D3D12_RESOURCE_STATES state_after)
{
    transition_sub_resource_state(list, resource, D3D12_RESOURCE_BARRIER_ALL_SUBRESOURCES,
            state_before, state_after);
}

static unsigned int format_size(DXGI_FORMAT format)
{
    switch (format)
    {
        case DXGI_FORMAT_R32G32B32A32_FLOAT:
        case DXGI_FORMAT_R32G32B32A32_UINT:
        case DXGI_FORMAT_R32G32B32A32_SINT:
        case DXGI_FORMAT_R8G8_UNORM:
            return 16;
        case DXGI_FORMAT_R16G16B16A16_TYPELESS:
        case DXGI_FORMAT_D32_FLOAT_S8X24_UINT:
            return 8;
        case DXGI_FORMAT_R32_TYPELESS:
        case DXGI_FORMAT_D32_FLOAT:
        case DXGI_FORMAT_R32_FLOAT:
        case DXGI_FORMAT_R32_UINT:
        case DXGI_FORMAT_R32_SINT:
        case DXGI_FORMAT_R16G16_FLOAT:
        case DXGI_FORMAT_R16G16_UNORM:
        case DXGI_FORMAT_R16G16_UINT:
        case DXGI_FORMAT_R11G11B10_FLOAT:
        case DXGI_FORMAT_R8G8B8A8_TYPELESS:
        case DXGI_FORMAT_R8G8B8A8_UNORM:
        case DXGI_FORMAT_R8G8B8A8_UNORM_SRGB:
        case DXGI_FORMAT_R8G8B8A8_UINT:
        case DXGI_FORMAT_B8G8R8A8_UNORM:
        case DXGI_FORMAT_D24_UNORM_S8_UINT:
            return 4;
        case DXGI_FORMAT_R16_FLOAT:
        case DXGI_FORMAT_R16_UNORM:
        case DXGI_FORMAT_R16_UINT:
            return 2;
        case DXGI_FORMAT_UNKNOWN:
        case DXGI_FORMAT_A8_UNORM:
        case DXGI_FORMAT_R8_TYPELESS:
        case DXGI_FORMAT_R8_UINT:
        case DXGI_FORMAT_R8_UNORM:
            return 1;

        case DXGI_FORMAT_BC2_UNORM:
        case DXGI_FORMAT_BC2_UNORM_SRGB:
        case DXGI_FORMAT_BC3_UNORM:
        case DXGI_FORMAT_BC3_UNORM_SRGB:
        case DXGI_FORMAT_BC5_UNORM:
        case DXGI_FORMAT_BC5_SNORM:
        case DXGI_FORMAT_BC6H_UF16:
        case DXGI_FORMAT_BC6H_SF16:
        case DXGI_FORMAT_BC7_UNORM:
        case DXGI_FORMAT_BC7_UNORM_SRGB:
            return 16;
        case DXGI_FORMAT_BC1_UNORM:
        case DXGI_FORMAT_BC1_UNORM_SRGB:
        case DXGI_FORMAT_BC4_UNORM:
        case DXGI_FORMAT_BC4_SNORM:
            return 8;
        case DXGI_FORMAT_B5G6R5_UNORM:
        case DXGI_FORMAT_B5G5R5A1_UNORM:
        case DXGI_FORMAT_B4G4R4A4_UNORM:
            return 2;

        default:
            trace("Unhandled format %#x.\n", format);
            return 1;
    }
}

static inline unsigned int format_block_width(DXGI_FORMAT format)
{
    switch (format)
    {
        case DXGI_FORMAT_BC1_UNORM:
        case DXGI_FORMAT_BC1_UNORM_SRGB:
        case DXGI_FORMAT_BC4_UNORM:
        case DXGI_FORMAT_BC4_SNORM:
        case DXGI_FORMAT_BC2_UNORM:
        case DXGI_FORMAT_BC2_UNORM_SRGB:
        case DXGI_FORMAT_BC3_UNORM:
        case DXGI_FORMAT_BC3_UNORM_SRGB:
        case DXGI_FORMAT_BC5_UNORM:
        case DXGI_FORMAT_BC5_SNORM:
        case DXGI_FORMAT_BC6H_UF16:
        case DXGI_FORMAT_BC6H_SF16:
        case DXGI_FORMAT_BC7_UNORM:
        case DXGI_FORMAT_BC7_UNORM_SRGB:
            return 4;
        default:
            return 1;
    }
}

static inline unsigned int format_block_height(DXGI_FORMAT format)
{
    switch (format)
    {
        case DXGI_FORMAT_BC1_UNORM:
        case DXGI_FORMAT_BC1_UNORM_SRGB:
        case DXGI_FORMAT_BC4_UNORM:
        case DXGI_FORMAT_BC4_SNORM:
        case DXGI_FORMAT_BC2_UNORM:
        case DXGI_FORMAT_BC2_UNORM_SRGB:
        case DXGI_FORMAT_BC3_UNORM:
        case DXGI_FORMAT_BC3_UNORM_SRGB:
        case DXGI_FORMAT_BC5_UNORM:
        case DXGI_FORMAT_BC5_SNORM:
        case DXGI_FORMAT_BC6H_UF16:
        case DXGI_FORMAT_BC6H_SF16:
        case DXGI_FORMAT_BC7_UNORM:
        case DXGI_FORMAT_BC7_UNORM_SRGB:
            return 4;
        default:
            return 1;
    }
}

static inline unsigned int format_plane_count(DXGI_FORMAT format)
{
    switch (format)
    {
        case DXGI_FORMAT_R32G8X24_TYPELESS:
        case DXGI_FORMAT_D32_FLOAT_S8X24_UINT:
        case DXGI_FORMAT_R32_FLOAT_X8X24_TYPELESS:
        case DXGI_FORMAT_X32_TYPELESS_G8X24_UINT:
        case DXGI_FORMAT_D24_UNORM_S8_UINT:
        case DXGI_FORMAT_R24G8_TYPELESS:
        case DXGI_FORMAT_R24_UNORM_X8_TYPELESS:
        case DXGI_FORMAT_X24_TYPELESS_G8_UINT:
        case DXGI_FORMAT_NV12:
        case DXGI_FORMAT_P010:
        case DXGI_FORMAT_P016:
        case DXGI_FORMAT_NV11:
            return 2;
        default:
            return 1;
    }
}

static inline DXGI_FORMAT format_get_subresource_plane_format(DXGI_FORMAT format, unsigned int plane_idx,
        unsigned int plane_count)
{
    assert(plane_idx < plane_count && plane_count >= 1 && plane_count <= 2);

    if (plane_count < 2)
        return format;

    return plane_idx ? DXGI_FORMAT_R8_TYPELESS : DXGI_FORMAT_R32_TYPELESS;
}

struct d3d12_resource_readback
{
    struct resource_readback rb;
    ID3D12Resource *resource;
};

#define RESOURCE_STATE_DO_NOT_CHANGE (~0u)

static void get_resource_readback_with_command_list_and_states(ID3D12Resource *resource, unsigned int sub_resource,
        struct d3d12_resource_readback *rb, ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *command_list,
        D3D12_RESOURCE_STATES initial_state, D3D12_RESOURCE_STATES final_state)
{
    D3D12_HEAP_PROPERTIES heap_properties;
    D3D12_RESOURCE_STATES readback_state;
    D3D12_RESOURCE_DESC resource_desc;
    ID3D12Resource *src_resource;
    D3D12_RANGE read_range;
    unsigned int miplevel;
    ID3D12Device *device;
    uint64_t buffer_size;
    HRESULT hr;
    bool is_ms;

    hr = ID3D12Resource_GetDevice(resource, &IID_ID3D12Device, (void **)&device);
    assert_that(hr == S_OK, "Failed to get device, hr %#x.\n", hr);

    resource_desc = ID3D12Resource_GetDesc(resource);

    miplevel = sub_resource % resource_desc.MipLevels;
    rb->rb.width = max(1, resource_desc.Width >> miplevel);
    rb->rb.width = align(rb->rb.width, format_block_width(resource_desc.Format));
    rb->rb.height = max(1, resource_desc.Height >> miplevel);
    rb->rb.height = align(rb->rb.height, format_block_height(resource_desc.Format));
    rb->rb.depth = resource_desc.Dimension == D3D12_RESOURCE_DIMENSION_TEXTURE3D
            ? max(1, resource_desc.DepthOrArraySize >> miplevel) : 1;
    rb->rb.row_pitch = rb->rb.width * format_size(resource_desc.Format);
    if (resource_desc.Dimension != D3D12_RESOURCE_DIMENSION_BUFFER)
        rb->rb.row_pitch = align(rb->rb.row_pitch, D3D12_TEXTURE_DATA_PITCH_ALIGNMENT);
    rb->rb.data = NULL;

    is_ms = resource_desc.Dimension != D3D12_RESOURCE_DIMENSION_BUFFER && resource_desc.SampleDesc.Count > 1;
    readback_state = is_ms ? D3D12_RESOURCE_STATE_RESOLVE_SOURCE : D3D12_RESOURCE_STATE_COPY_SOURCE;

    if (initial_state != RESOURCE_STATE_DO_NOT_CHANGE)
        transition_sub_resource_state(command_list, resource, sub_resource, initial_state, readback_state);

    src_resource = resource;
    if (is_ms)
    {
        memset(&heap_properties, 0, sizeof(heap_properties));
        heap_properties.Type = D3D12_HEAP_TYPE_DEFAULT;

        resource_desc.Alignment = 0;
        resource_desc.DepthOrArraySize = 1;
        resource_desc.SampleDesc.Count = 1;
        resource_desc.SampleDesc.Quality = 0;
        hr = ID3D12Device_CreateCommittedResource(device, &heap_properties, D3D12_HEAP_FLAG_NONE,
                &resource_desc, D3D12_RESOURCE_STATE_RESOLVE_DEST, NULL,
                &IID_ID3D12Resource, (void **)&src_resource);
        assert_that(hr == S_OK, "Failed to create resource, hr %#x.\n", hr);

        ID3D12GraphicsCommandList_ResolveSubresource(command_list,
                src_resource, 0, resource, sub_resource, resource_desc.Format);
        transition_resource_state(command_list, src_resource,
                D3D12_RESOURCE_STATE_RESOLVE_DEST, D3D12_RESOURCE_STATE_COPY_SOURCE);

        sub_resource = 0;
    }

    buffer_size = rb->rb.row_pitch * rb->rb.height * rb->rb.depth;
    rb->resource = create_readback_buffer(device, buffer_size);

    if (resource_desc.Dimension == D3D12_RESOURCE_DIMENSION_BUFFER)
    {
        ID3D12GraphicsCommandList_CopyResource(command_list, rb->resource, resource);
    }
    else
    {
        D3D12_TEXTURE_COPY_LOCATION dst_location, src_location;

        dst_location.pResource = rb->resource;
        dst_location.Type = D3D12_TEXTURE_COPY_TYPE_PLACED_FOOTPRINT;
        dst_location.PlacedFootprint.Offset = 0;
        dst_location.PlacedFootprint.Footprint.Format = resource_desc.Format;
        dst_location.PlacedFootprint.Footprint.Width = rb->rb.width;
        dst_location.PlacedFootprint.Footprint.Height = rb->rb.height;
        dst_location.PlacedFootprint.Footprint.Depth = rb->rb.depth;
        dst_location.PlacedFootprint.Footprint.RowPitch = rb->rb.row_pitch;

        src_location.pResource = src_resource;
        src_location.Type = D3D12_TEXTURE_COPY_TYPE_SUBRESOURCE_INDEX;
        src_location.SubresourceIndex = sub_resource;

        ID3D12GraphicsCommandList_CopyTextureRegion(command_list, &dst_location, 0, 0, 0, &src_location, NULL);
    }

    if (final_state != RESOURCE_STATE_DO_NOT_CHANGE)
        transition_sub_resource_state(command_list, resource, sub_resource, readback_state, final_state);

    hr = ID3D12GraphicsCommandList_Close(command_list);
    assert_that(hr == S_OK, "Failed to close command list, hr %#x.\n", hr);

    exec_command_list(queue, command_list);
    wait_queue_idle(device, queue);
    ID3D12Device_Release(device);

    if (src_resource != resource)
        ID3D12Resource_Release(src_resource);

    read_range.Begin = 0;
    read_range.End = buffer_size;
    hr = ID3D12Resource_Map(rb->resource, 0, &read_range, &rb->rb.data);
    assert_that(hr == S_OK, "Failed to map readback buffer, hr %#x.\n", hr);
}

static void get_resource_readback_with_command_list(ID3D12Resource *resource, unsigned int sub_resource,
        struct d3d12_resource_readback *rb, ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *command_list)
{
    return get_resource_readback_with_command_list_and_states(resource, sub_resource, rb, queue, command_list,
            RESOURCE_STATE_DO_NOT_CHANGE, RESOURCE_STATE_DO_NOT_CHANGE);
}

static void release_resource_readback(struct d3d12_resource_readback *rb)
{
    D3D12_RANGE range = {0, 0};
    ID3D12Resource_Unmap(rb->resource, 0, &range);
    ID3D12Resource_Release(rb->resource);
}

#define check_sub_resource_uint(a, b, c, d, e, f) check_sub_resource_uint_(__FILE__, __LINE__, a, b, c, d, e, f)
static inline void check_sub_resource_uint_(const char *file, unsigned int line, ID3D12Resource *texture,
        unsigned int sub_resource_idx, ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *command_list,
        unsigned int expected, unsigned int max_diff)
{
    struct d3d12_resource_readback rb;

    get_resource_readback_with_command_list(texture, sub_resource_idx, &rb, queue, command_list);
    check_readback_data_uint_(file, line, &rb.rb, NULL, expected, max_diff);
    release_resource_readback(&rb);
}

#define check_sub_resource_uint_with_box(a, b, c, d, e, f, g) \
        check_sub_resource_uint_with_box_(__FILE__, __LINE__, a, b, c, d, e, f, g)
static inline void check_sub_resource_uint_with_box_(const char *file, unsigned int line, ID3D12Resource *texture,
        unsigned int sub_resource_idx, ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *command_list,
        const D3D12_BOX *box, unsigned int expected, unsigned int max_diff)
{
    struct d3d12_resource_readback rb;

    get_resource_readback_with_command_list(texture, sub_resource_idx, &rb, queue, command_list);
    check_readback_data_uint_(file, line, &rb.rb, box, expected, max_diff);
    release_resource_readback(&rb);
}

#define check_sub_resource_vec4(a, b, c, d, e, f) check_sub_resource_vec4_(__FILE__, __LINE__, a, b, c, d, e, f)
static inline void check_sub_resource_vec4_(const char *file, unsigned int line, ID3D12Resource *texture,
        unsigned int sub_resource_idx, ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *command_list,
        const struct vec4 *expected, unsigned int max_diff)
{
    struct d3d12_resource_readback rb;

    get_resource_readback_with_command_list(texture, sub_resource_idx, &rb, queue, command_list);
    check_readback_data_vec_(file, line, &rb.rb, NULL, expected, max_diff, 4);
    release_resource_readback(&rb);
}

#define create_default_buffer(a, b, c, d) create_default_buffer_(__FILE__, __LINE__, a, b, c, d)
static inline ID3D12Resource *create_default_buffer_(const char *file, unsigned int line, ID3D12Device *device,
        size_t size, D3D12_RESOURCE_FLAGS resource_flags, D3D12_RESOURCE_STATES initial_resource_state)
{
    return create_buffer_(file, line, device, D3D12_HEAP_TYPE_DEFAULT, size,
            resource_flags, initial_resource_state);
}

static ID3D12Resource *create_default_texture_(const char *file, unsigned int line, ID3D12Device *device,
        D3D12_RESOURCE_DIMENSION dimension, unsigned int width, unsigned int height,
        unsigned int depth_or_array_size, unsigned int miplevel_count, unsigned int sample_count,
        DXGI_FORMAT format, D3D12_RESOURCE_FLAGS flags, D3D12_RESOURCE_STATES initial_state)
{
    D3D12_HEAP_PROPERTIES heap_properties;
    D3D12_RESOURCE_DESC resource_desc;
    ID3D12Resource *texture;
    HRESULT hr;

    assert(dimension != D3D12_RESOURCE_DIMENSION_BUFFER);

    memset(&heap_properties, 0, sizeof(heap_properties));
    heap_properties.Type = D3D12_HEAP_TYPE_DEFAULT;

    memset(&resource_desc, 0, sizeof(resource_desc));
    resource_desc.Dimension = dimension;
    resource_desc.Width = width;
    resource_desc.Height = height;
    resource_desc.DepthOrArraySize = depth_or_array_size;
    resource_desc.MipLevels = miplevel_count;
    resource_desc.Format = format;
    resource_desc.SampleDesc.Count = max(sample_count, 1);
    resource_desc.Flags = flags;
    hr = ID3D12Device_CreateCommittedResource(device, &heap_properties, D3D12_HEAP_FLAG_NONE,
            &resource_desc, initial_state, NULL, &IID_ID3D12Resource, (void **)&texture);
    ok_(file, line)(SUCCEEDED(hr), "Failed to create texture, hr %#x.\n", hr);

    return texture;
}

#define create_default_texture(a, b, c, d, e, f) create_default_texture2d_(__FILE__, __LINE__, a, b, c, 1, 1, d, e, f)
#define create_default_texture2d(a, b, c, d, e, f, g, h) \
        create_default_texture2d_(__FILE__, __LINE__, a, b, c, d, e, f, g, h)
static inline ID3D12Resource *create_default_texture2d_(const char *file, unsigned int line, ID3D12Device *device,
        unsigned int width, unsigned int height, unsigned int array_size, unsigned int miplevel_count,
        DXGI_FORMAT format, D3D12_RESOURCE_FLAGS flags, D3D12_RESOURCE_STATES initial_state)
{
    return create_default_texture_(file, line, device, D3D12_RESOURCE_DIMENSION_TEXTURE2D,
            width, height, array_size, miplevel_count, 1, format, flags, initial_state);
}

#define create_default_texture3d(a, b, c, d, e, f, g, h) \
        create_default_texture3d_(__FILE__, __LINE__, a, b, c, d, e, f, g, h)
static inline ID3D12Resource *create_default_texture3d_(const char *file, unsigned int line, ID3D12Device *device,
        unsigned int width, unsigned int height, unsigned int depth, unsigned int miplevel_count,
        DXGI_FORMAT format, D3D12_RESOURCE_FLAGS flags, D3D12_RESOURCE_STATES initial_state)
{
    return create_default_texture_(file, line, device, D3D12_RESOURCE_DIMENSION_TEXTURE3D,
            width, height, depth, miplevel_count, 1, format, flags, initial_state);
}

static void copy_sub_resource_data(const D3D12_MEMCPY_DEST *dst, const D3D12_SUBRESOURCE_DATA *src,
        unsigned int row_count, unsigned int slice_count, size_t row_size)
{
    const BYTE *src_slice_ptr;
    BYTE *dst_slice_ptr;
    unsigned int z, y;

    for (z = 0; z < slice_count; ++z)
    {
        dst_slice_ptr = (BYTE *)dst->pData + z * dst->SlicePitch;
        src_slice_ptr = (const BYTE*)src->pData + z * src->SlicePitch;
        for (y = 0; y < row_count; ++y)
            memcpy(dst_slice_ptr + y * dst->RowPitch, src_slice_ptr + y * src->RowPitch, row_size);
    }
}

#define upload_texture_data_with_states(a, b, c, d, e, f, g) \
        upload_texture_data_with_states_(__FILE__, __LINE__, a, b, c, d, e, f, g)
static inline void upload_texture_data_with_states_(const char *file, unsigned int line,
        ID3D12Resource *texture, const D3D12_SUBRESOURCE_DATA *data, unsigned int sub_resource_count,
        ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *command_list,
        D3D12_RESOURCE_STATES initial_state, D3D12_RESOURCE_STATES final_state)
{
    D3D12_TEXTURE_COPY_LOCATION dst_location, src_location;
    D3D12_PLACED_SUBRESOURCE_FOOTPRINT *layouts;
    uint64_t *row_sizes, required_size;
    D3D12_RESOURCE_DESC resource_desc;
    ID3D12Resource *upload_buffer;
    D3D12_MEMCPY_DEST dst_data;
    ID3D12Device *device;
    UINT *row_counts;
    unsigned int i;
    HRESULT hr;
    void *ptr;

    layouts = calloc(sub_resource_count, sizeof(*layouts));
    ok(layouts, "Failed to allocate memory.\n");
    row_counts = calloc(sub_resource_count, sizeof(*row_counts));
    ok(row_counts, "Failed to allocate memory.\n");
    row_sizes = calloc(sub_resource_count, sizeof(*row_sizes));
    ok(row_sizes, "Failed to allocate memory.\n");

    resource_desc = ID3D12Resource_GetDesc(texture);
    hr = ID3D12Resource_GetDevice(texture, &IID_ID3D12Device, (void **)&device);
    ok_(file, line)(SUCCEEDED(hr), "Failed to get device, hr %#x.\n", hr);

    ID3D12Device_GetCopyableFootprints(device, &resource_desc, 0, sub_resource_count,
            0, layouts, row_counts, row_sizes, &required_size);

    upload_buffer = create_upload_buffer_(file, line, device, required_size, NULL);

    hr = ID3D12Resource_Map(upload_buffer, 0, NULL, (void **)&ptr);
    ok_(file, line)(SUCCEEDED(hr), "Failed to map upload buffer, hr %#x.\n", hr);
    for (i = 0; i < sub_resource_count; ++i)
    {
        dst_data.pData = (BYTE *)ptr + layouts[i].Offset;
        dst_data.RowPitch = layouts[i].Footprint.RowPitch;
        dst_data.SlicePitch = layouts[i].Footprint.RowPitch * row_counts[i];
        copy_sub_resource_data(&dst_data, &data[i],
                row_counts[i], layouts[i].Footprint.Depth, row_sizes[i]);
    }
    ID3D12Resource_Unmap(upload_buffer, 0, NULL);

    if (resource_desc.Dimension == D3D12_RESOURCE_DIMENSION_BUFFER)
    {
        if (initial_state != RESOURCE_STATE_DO_NOT_CHANGE)
            transition_resource_state(command_list, texture, initial_state, D3D12_RESOURCE_STATE_COPY_DEST);

        ID3D12GraphicsCommandList_CopyResource(command_list, texture, upload_buffer);

        if (final_state != RESOURCE_STATE_DO_NOT_CHANGE)
            transition_resource_state(command_list, texture, D3D12_RESOURCE_STATE_COPY_DEST, final_state);
    }
    else
    {
        for (i = 0; i < sub_resource_count; ++i)
        {
            dst_location.pResource = texture;
            dst_location.Type = D3D12_TEXTURE_COPY_TYPE_SUBRESOURCE_INDEX;
            dst_location.SubresourceIndex = i;

            src_location.pResource = upload_buffer;
            src_location.Type = D3D12_TEXTURE_COPY_TYPE_PLACED_FOOTPRINT;
            src_location.PlacedFootprint = layouts[i];

            if (initial_state != RESOURCE_STATE_DO_NOT_CHANGE)
                transition_sub_resource_state(command_list, texture, i, initial_state, D3D12_RESOURCE_STATE_COPY_DEST);

            ID3D12GraphicsCommandList_CopyTextureRegion(command_list,
                    &dst_location, 0, 0, 0, &src_location, NULL);

            if (final_state != RESOURCE_STATE_DO_NOT_CHANGE)
                transition_sub_resource_state(command_list, texture, i, D3D12_RESOURCE_STATE_COPY_DEST, final_state);
        }
    }

    hr = ID3D12GraphicsCommandList_Close(command_list);
    ok_(file, line)(SUCCEEDED(hr), "Failed to close command list, hr %#x.\n", hr);

    exec_command_list(queue, command_list);
    wait_queue_idle(device, queue);

    ID3D12Resource_Release(upload_buffer);
    ID3D12Device_Release(device);

    free(layouts);
    free(row_counts);
    free(row_sizes);
}

#define upload_texture_data(a, b, c, d, e) upload_texture_data_(__FILE__, __LINE__, a, b, c, d, e)
static inline void upload_texture_data_(const char *file, unsigned int line, ID3D12Resource *texture,
        const D3D12_SUBRESOURCE_DATA *data, unsigned int sub_resource_count,
        ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *command_list)
{
    return upload_texture_data_with_states_(file, line, texture, data, sub_resource_count, queue, command_list,
            RESOURCE_STATE_DO_NOT_CHANGE, RESOURCE_STATE_DO_NOT_CHANGE);
}

#define upload_buffer_data_with_states(a, b, c, d, e, f, g, h) \
        upload_buffer_data_with_states_(__FILE__, __LINE__, a, b, c, d, e, f, g, h)
static inline void upload_buffer_data_with_states_(const char *file, unsigned int line,
        ID3D12Resource *buffer, size_t offset, size_t size, const void *data,
        ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *command_list,
        D3D12_RESOURCE_STATES initial_state, D3D12_RESOURCE_STATES final_state)
{
    ID3D12Resource *upload_buffer;
    ID3D12Device *device;
    HRESULT hr;

    hr = ID3D12Resource_GetDevice(buffer, &IID_ID3D12Device, (void **)&device);
    ok_(file, line)(SUCCEEDED(hr), "Failed to get device, hr %#x.\n", hr);

    upload_buffer = create_upload_buffer_(file, line, device, size, data);

    if (initial_state != RESOURCE_STATE_DO_NOT_CHANGE)
        transition_resource_state(command_list, buffer, initial_state, D3D12_RESOURCE_STATE_COPY_DEST);

    ID3D12GraphicsCommandList_CopyBufferRegion(command_list, buffer, offset,
            upload_buffer, 0, size);

    if (final_state != RESOURCE_STATE_DO_NOT_CHANGE)
        transition_resource_state(command_list, buffer, D3D12_RESOURCE_STATE_COPY_DEST, final_state);

    hr = ID3D12GraphicsCommandList_Close(command_list);
    ok_(file, line)(SUCCEEDED(hr), "Failed to close command list, hr %#x.\n", hr);
    exec_command_list(queue, command_list);
    wait_queue_idle(device, queue);

    ID3D12Resource_Release(upload_buffer);
    ID3D12Device_Release(device);
}

#define upload_buffer_data(a, b, c, d, e, f) upload_buffer_data_(__FILE__, __LINE__, a, b, c, d, e, f)
static inline void upload_buffer_data_(const char *file, unsigned int line, ID3D12Resource *buffer, size_t offset,
        size_t size, const void *data, ID3D12CommandQueue *queue, ID3D12GraphicsCommandList *command_list)
{
    return upload_buffer_data_with_states_(file, line, buffer, offset, size, data, queue, command_list,
            RESOURCE_STATE_DO_NOT_CHANGE, RESOURCE_STATE_DO_NOT_CHANGE);
}

static HRESULT create_root_signature(ID3D12Device *device, const D3D12_ROOT_SIGNATURE_DESC *desc,
        ID3D12RootSignature **root_signature)
{
    ID3DBlob *blob;
    HRESULT hr;

    if (FAILED(hr = D3D12SerializeRootSignature(desc, D3D_ROOT_SIGNATURE_VERSION_1_0, &blob, NULL)))
        return hr;

    hr = ID3D12Device_CreateRootSignature(device, 0, ID3D10Blob_GetBufferPointer(blob),
            ID3D10Blob_GetBufferSize(blob), &IID_ID3D12RootSignature, (void **)root_signature);
    ID3D10Blob_Release(blob);
    return hr;
}

#define create_empty_root_signature(device, flags) create_empty_root_signature_(__FILE__, __LINE__, device, flags)
static ID3D12RootSignature *create_empty_root_signature_(const char *file,
        unsigned int line, ID3D12Device *device, D3D12_ROOT_SIGNATURE_FLAGS flags)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    ID3D12RootSignature *root_signature = NULL;
    HRESULT hr;

    root_signature_desc.NumParameters = 0;
    root_signature_desc.pParameters = NULL;
    root_signature_desc.NumStaticSamplers = 0;
    root_signature_desc.pStaticSamplers = NULL;
    root_signature_desc.Flags = flags;
    hr = create_root_signature(device, &root_signature_desc, &root_signature);
    ok_(file, line)(SUCCEEDED(hr), "Failed to create root signature, hr %#x.\n", hr);

    return root_signature;
}

#define create_32bit_constants_root_signature(a, b, c, e) \
        create_32bit_constants_root_signature_(__FILE__, __LINE__, a, b, c, e, 0)
static inline ID3D12RootSignature *create_32bit_constants_root_signature_(const char *file,
        unsigned int line, ID3D12Device *device, unsigned int reg_idx, unsigned int element_count,
        D3D12_SHADER_VISIBILITY shader_visibility, D3D12_ROOT_SIGNATURE_FLAGS flags)
{
    D3D12_ROOT_SIGNATURE_DESC root_signature_desc;
    ID3D12RootSignature *root_signature = NULL;
    D3D12_ROOT_PARAMETER root_parameter;
    HRESULT hr;

    root_parameter.ParameterType = D3D12_ROOT_PARAMETER_TYPE_32BIT_CONSTANTS;
    root_parameter.Constants.ShaderRegister = reg_idx;
    root_parameter.Constants.RegisterSpace = 0;
    root_parameter.Constants.Num32BitValues = element_count;
    root_parameter.ShaderVisibility = shader_visibility;

    memset(&root_signature_desc, 0, sizeof(root_signature_desc));
    root_signature_desc.NumParameters = 1;
    root_signature_desc.pParameters = &root_parameter;
    root_signature_desc.Flags = flags;
    hr = create_root_signature(device, &root_signature_desc, &root_signature);
    ok_(file, line)(SUCCEEDED(hr), "Failed to create root signature, hr %#x.\n", hr);

    return root_signature;
}

static void init_pipeline_state_desc(D3D12_GRAPHICS_PIPELINE_STATE_DESC *desc,
        ID3D12RootSignature *root_signature, DXGI_FORMAT rt_format,
        const D3D12_SHADER_BYTECODE *vs, const D3D12_SHADER_BYTECODE *ps,
        const D3D12_INPUT_LAYOUT_DESC *input_layout)
{
    static const DWORD vs_code[] =
    {
#if 0
        void main(uint id : SV_VertexID, out float4 position : SV_Position)
        {
            float2 coords = float2((id << 1) & 2, id & 2);
            position = float4(coords * float2(2, -2) + float2(-1, 1), 0, 1);
        }
#endif
        0x43425844, 0xf900d25e, 0x68bfefa7, 0xa63ac0a7, 0xa476af7a, 0x00000001, 0x0000018c, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000006, 0x00000001, 0x00000000, 0x00000101, 0x565f5653, 0x65747265, 0x00444978,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000001, 0x00000003,
        0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69, 0x58454853, 0x000000f0, 0x00010050,
        0x0000003c, 0x0100086a, 0x04000060, 0x00101012, 0x00000000, 0x00000006, 0x04000067, 0x001020f2,
        0x00000000, 0x00000001, 0x02000068, 0x00000001, 0x0b00008c, 0x00100012, 0x00000000, 0x00004001,
        0x00000001, 0x00004001, 0x00000001, 0x0010100a, 0x00000000, 0x00004001, 0x00000000, 0x07000001,
        0x00100042, 0x00000000, 0x0010100a, 0x00000000, 0x00004001, 0x00000002, 0x05000056, 0x00100032,
        0x00000000, 0x00100086, 0x00000000, 0x0f000032, 0x00102032, 0x00000000, 0x00100046, 0x00000000,
        0x00004002, 0x40000000, 0xc0000000, 0x00000000, 0x00000000, 0x00004002, 0xbf800000, 0x3f800000,
        0x00000000, 0x00000000, 0x08000036, 0x001020c2, 0x00000000, 0x00004002, 0x00000000, 0x00000000,
        0x00000000, 0x3f800000, 0x0100003e,
    };
    static const DWORD ps_code[] =
    {
#if 0
        void main(const in float4 position : SV_Position, out float4 target : SV_Target0)
        {
            target = float4(0.0f, 1.0f, 0.0f, 1.0f);
        }
#endif
        0x43425844, 0x8a4a8140, 0x5eba8e0b, 0x714e0791, 0xb4b8eed2, 0x00000001, 0x000000d8, 0x00000003,
        0x0000002c, 0x00000060, 0x00000094, 0x4e475349, 0x0000002c, 0x00000001, 0x00000008, 0x00000020,
        0x00000000, 0x00000001, 0x00000003, 0x00000000, 0x0000000f, 0x505f5653, 0x7469736f, 0x006e6f69,
        0x4e47534f, 0x0000002c, 0x00000001, 0x00000008, 0x00000020, 0x00000000, 0x00000000, 0x00000003,
        0x00000000, 0x0000000f, 0x545f5653, 0x65677261, 0xabab0074, 0x58454853, 0x0000003c, 0x00000050,
        0x0000000f, 0x0100086a, 0x03000065, 0x001020f2, 0x00000000, 0x08000036, 0x001020f2, 0x00000000,
        0x00004002, 0x00000000, 0x3f800000, 0x00000000, 0x3f800000, 0x0100003e,
    };

    memset(desc, 0, sizeof(*desc));
    desc->pRootSignature = root_signature;
    if (vs)
        desc->VS = *vs;
    else
        desc->VS = shader_bytecode(vs_code, sizeof(vs_code));
    if (ps)
        desc->PS = *ps;
    else
        desc->PS = shader_bytecode(ps_code, sizeof(ps_code));
    desc->StreamOutput.RasterizedStream = 0;
    desc->BlendState.RenderTarget[0].RenderTargetWriteMask = D3D12_COLOR_WRITE_ENABLE_ALL;
    desc->RasterizerState.FillMode = D3D12_FILL_MODE_SOLID;
    desc->RasterizerState.CullMode = D3D12_CULL_MODE_BACK;
    if (input_layout)
        desc->InputLayout = *input_layout;
    desc->SampleMask = ~(UINT)0;
    desc->PrimitiveTopologyType = D3D12_PRIMITIVE_TOPOLOGY_TYPE_TRIANGLE;
    if (rt_format)
    {
        desc->NumRenderTargets = 1;
        desc->RTVFormats[0] = rt_format;
    }
    desc->SampleDesc.Count = 1;
}

#define create_pipeline_state(a, b, c, d, e, f) create_pipeline_state_(__FILE__, __LINE__, a, b, c, d, e, f)
static ID3D12PipelineState *create_pipeline_state_(const char *file, unsigned int line, ID3D12Device *device,
        ID3D12RootSignature *root_signature, DXGI_FORMAT rt_format, const D3D12_SHADER_BYTECODE *vs,
        const D3D12_SHADER_BYTECODE *ps, const D3D12_INPUT_LAYOUT_DESC *input_layout)
{
    D3D12_GRAPHICS_PIPELINE_STATE_DESC pipeline_state_desc;
    ID3D12PipelineState *pipeline_state = NULL;
    HRESULT hr;

    init_pipeline_state_desc(&pipeline_state_desc, root_signature, rt_format, vs, ps, input_layout);
    hr = ID3D12Device_CreateGraphicsPipelineState(device, &pipeline_state_desc,
            &IID_ID3D12PipelineState, (void **)&pipeline_state);
    ok_(file, line)(SUCCEEDED(hr), "Failed to create graphics pipeline state, hr %#x.\n", hr);

    return SUCCEEDED(hr) ? pipeline_state : NULL;
}

#define create_compute_pipeline_state(a, b, c) create_compute_pipeline_state_(__FILE__, __LINE__, a, b, c)
static inline ID3D12PipelineState *create_compute_pipeline_state_(const char *file, unsigned int line,
        ID3D12Device *device, ID3D12RootSignature *root_signature, const D3D12_SHADER_BYTECODE cs)
{
    D3D12_COMPUTE_PIPELINE_STATE_DESC pipeline_state_desc;
    ID3D12PipelineState *pipeline_state = NULL;
    HRESULT hr;

    memset(&pipeline_state_desc, 0, sizeof(pipeline_state_desc));
    pipeline_state_desc.pRootSignature = root_signature;
    pipeline_state_desc.CS = cs;
    pipeline_state_desc.NodeMask = 0;
    pipeline_state_desc.Flags = D3D12_PIPELINE_STATE_FLAG_NONE;
    hr = ID3D12Device_CreateComputePipelineState(device, &pipeline_state_desc,
            &IID_ID3D12PipelineState, (void **)&pipeline_state);
    ok_(file, line)(SUCCEEDED(hr), "Failed to create compute pipeline state, hr %#x.\n", hr);

    return pipeline_state;
}

struct test_context_desc
{
    unsigned int rt_width, rt_height, rt_array_size;
    DXGI_FORMAT rt_format;
    DXGI_SAMPLE_DESC sample_desc;
    bool check_multisampling;
    unsigned int rt_descriptor_count;
    unsigned int root_signature_flags;
    bool no_render_target;
    bool no_root_signature;
    bool no_pipeline;
    const D3D12_SHADER_BYTECODE *vs;
    const D3D12_SHADER_BYTECODE *ps;
};

struct test_context
{
    ID3D12Device *device;
    ID3D12Device2 *device2;

    ID3D12CommandQueue *queue;
    ID3D12CommandAllocator *allocator;
    ID3D12GraphicsCommandList *list;
    ID3D12GraphicsCommandList1 *list1;

    D3D12_RESOURCE_DESC render_target_desc;
    ID3D12Resource *render_target;

    ID3D12DescriptorHeap *rtv_heap;
    D3D12_CPU_DESCRIPTOR_HANDLE rtv;

    ID3D12RootSignature *root_signature;
    ID3D12PipelineState *pipeline_state;

    D3D12_VIEWPORT viewport;
    RECT scissor_rect;
};

#define check_multisample_support(a, b, c) check_multisample_support_(__FILE__, __LINE__, a, b, c)
static unsigned int check_multisample_support_(const char *file, unsigned int line,
        ID3D12Device *device, DXGI_FORMAT format, unsigned int sample_count)
{
    D3D12_FEATURE_DATA_MULTISAMPLE_QUALITY_LEVELS format_support =
    {
        .Format = format,
        .SampleCount = sample_count,
    };
    HRESULT hr;

    hr = ID3D12Device_CheckFeatureSupport(device, D3D12_FEATURE_MULTISAMPLE_QUALITY_LEVELS,
            &format_support, sizeof(format_support));
    ok_(file, line)(hr == S_OK, "Cannot check feature support, hr %#x.\n", hr);

    return format_support.NumQualityLevels;
}

#define create_render_target(context, a, b, c) create_render_target_(__FILE__, __LINE__, context, a, b, c)
static void create_render_target_(const char *file, unsigned int line, struct test_context *context,
        const struct test_context_desc *desc, ID3D12Resource **render_target, const D3D12_CPU_DESCRIPTOR_HANDLE *rtv)
{
    D3D12_HEAP_PROPERTIES heap_properties;
    D3D12_RESOURCE_DESC resource_desc;
    unsigned int quality_level_count;
    D3D12_CLEAR_VALUE clear_value;
    HRESULT hr;

    memset(&heap_properties, 0, sizeof(heap_properties));
    heap_properties.Type = D3D12_HEAP_TYPE_DEFAULT;

    resource_desc.Dimension = D3D12_RESOURCE_DIMENSION_TEXTURE2D;
    resource_desc.Alignment = 0;
    resource_desc.Width = desc && desc->rt_width ? desc->rt_width : 32;
    resource_desc.Height = desc && desc->rt_height ? desc->rt_height : 32;
    resource_desc.DepthOrArraySize = desc && desc->rt_array_size ? desc->rt_array_size : 1;
    resource_desc.MipLevels = 1;
    resource_desc.Format = desc && desc->rt_format ? desc->rt_format : DXGI_FORMAT_R8G8B8A8_UNORM;
    resource_desc.SampleDesc.Count = desc && desc->sample_desc.Count ? desc->sample_desc.Count : 1;
    resource_desc.SampleDesc.Quality = desc && desc->sample_desc.Count ? desc->sample_desc.Quality : 0;
    resource_desc.Layout = D3D12_TEXTURE_LAYOUT_UNKNOWN;
    resource_desc.Flags = D3D12_RESOURCE_FLAG_ALLOW_RENDER_TARGET;
    clear_value.Format = resource_desc.Format;
    clear_value.Color[0] = 1.0f;
    clear_value.Color[1] = 1.0f;
    clear_value.Color[2] = 1.0f;
    clear_value.Color[3] = 1.0f;

    if (desc && desc->check_multisampling)
    {
        for (; resource_desc.SampleDesc.Count != 1; resource_desc.SampleDesc.Count /= 2)
        {
            if ((quality_level_count = check_multisample_support_(file, line,
                    context->device, resource_desc.Format, resource_desc.SampleDesc.Count)))
            {
                resource_desc.SampleDesc.Quality = min(resource_desc.SampleDesc.Quality, quality_level_count - 1);
                break;
            }
        }
    }

    hr = ID3D12Device_CreateCommittedResource(context->device,
            &heap_properties, D3D12_HEAP_FLAG_NONE, &resource_desc,
            D3D12_RESOURCE_STATE_RENDER_TARGET, &clear_value,
            &IID_ID3D12Resource, (void **)render_target);
    ok_(file, line)(hr == S_OK, "Failed to create texture, hr %#x.\n", hr);

    context->render_target_desc = resource_desc;

    if (rtv)
        ID3D12Device_CreateRenderTargetView(context->device, *render_target, NULL, *rtv);
}

#define init_test_context(context, desc) init_test_context_(__FILE__, __LINE__, context, desc)
static inline bool init_test_context_(const char *file, unsigned int line,
        struct test_context *context, const struct test_context_desc *desc)
{
    D3D12_DESCRIPTOR_HEAP_DESC rtv_heap_desc;
    ID3D12Device *device;
    HRESULT hr;

    memset(context, 0, sizeof(*context));

    if (!(context->device = create_device()))
    {
        skip_(file, line)("Failed to create device.\n");
        return false;
    }
    device = context->device;

    if (FAILED(hr = ID3D12Device_QueryInterface(device, &IID_ID3D12Device2, (void **)&context->device2)))
        context->device2 = NULL;

    context->queue = create_command_queue_(file, line, device,
            D3D12_COMMAND_LIST_TYPE_DIRECT, D3D12_COMMAND_QUEUE_PRIORITY_NORMAL);

    hr = ID3D12Device_CreateCommandAllocator(device, D3D12_COMMAND_LIST_TYPE_DIRECT,
            &IID_ID3D12CommandAllocator, (void **)&context->allocator);
    ok_(file, line)(SUCCEEDED(hr), "Failed to create command allocator, hr %#x.\n", hr);

    hr = ID3D12Device_CreateCommandList(device, 0, D3D12_COMMAND_LIST_TYPE_DIRECT,
            context->allocator, NULL, &IID_ID3D12GraphicsCommandList, (void **)&context->list);
    ok_(file, line)(SUCCEEDED(hr), "Failed to create command list, hr %#x.\n", hr);

    if (FAILED(hr = ID3D12GraphicsCommandList_QueryInterface(context->list,
            &IID_ID3D12GraphicsCommandList1, (void**)&context->list1)))
        context->list1 = NULL;

    if (desc && desc->no_render_target)
        return true;

    rtv_heap_desc.NumDescriptors = desc && desc->rt_descriptor_count ? desc->rt_descriptor_count : 1;
    rtv_heap_desc.Type = D3D12_DESCRIPTOR_HEAP_TYPE_RTV;
    rtv_heap_desc.Flags = D3D12_DESCRIPTOR_HEAP_FLAG_NONE;
    rtv_heap_desc.NodeMask = 0;
    hr = ID3D12Device_CreateDescriptorHeap(device, &rtv_heap_desc,
            &IID_ID3D12DescriptorHeap, (void **)&context->rtv_heap);
    ok_(file, line)(SUCCEEDED(hr), "Failed to create descriptor heap, hr %#x.\n", hr);

    context->rtv = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(context->rtv_heap);

    create_render_target_(file, line, context, desc, &context->render_target, &context->rtv);

    set_viewport(&context->viewport, 0.0f, 0.0f,
            context->render_target_desc.Width, context->render_target_desc.Height, 0.0f, 1.0f);
    set_rect(&context->scissor_rect, 0, 0,
            context->render_target_desc.Width, context->render_target_desc.Height);

    if (desc && desc->no_root_signature)
        return true;

    context->root_signature = create_empty_root_signature_(file, line,
            device, desc ? desc->root_signature_flags : D3D12_ROOT_SIGNATURE_FLAG_NONE);

    if (desc && desc->no_pipeline)
        return true;

    context->pipeline_state = create_pipeline_state_(file, line, device,
            context->root_signature, context->render_target_desc.Format,
            desc ? desc->vs : NULL, desc ? desc->ps : NULL, NULL);

    return true;
}

#define init_compute_test_context(context) init_compute_test_context_(__FILE__, __LINE__, context)
static inline bool init_compute_test_context_(const char *file, unsigned int line, struct test_context *context)
{
    ID3D12Device *device;
    HRESULT hr;

    memset(context, 0, sizeof(*context));

    if (!(context->device = create_device()))
    {
        skip_(file, line)("Failed to create device.\n");
        return false;
    }
    device = context->device;

    context->queue = create_command_queue_(file, line, device,
            D3D12_COMMAND_LIST_TYPE_COMPUTE, D3D12_COMMAND_QUEUE_PRIORITY_NORMAL);

    hr = ID3D12Device_CreateCommandAllocator(device, D3D12_COMMAND_LIST_TYPE_COMPUTE,
            &IID_ID3D12CommandAllocator, (void **)&context->allocator);
    ok_(file, line)(hr == S_OK, "Failed to create command allocator, hr %#x.\n", hr);

    hr = ID3D12Device_CreateCommandList(device, 0, D3D12_COMMAND_LIST_TYPE_COMPUTE,
            context->allocator, NULL, &IID_ID3D12GraphicsCommandList, (void **)&context->list);
    ok_(file, line)(hr == S_OK, "Failed to create command list, hr %#x.\n", hr);

    return true;
}

#define destroy_test_context(context) destroy_test_context_(__FILE__, __LINE__, context)
static inline void destroy_test_context_(const char *file, unsigned int line, struct test_context *context)
{
    ULONG refcount;

    if (context->pipeline_state)
        ID3D12PipelineState_Release(context->pipeline_state);
    if (context->root_signature)
        ID3D12RootSignature_Release(context->root_signature);

    if (context->rtv_heap)
        ID3D12DescriptorHeap_Release(context->rtv_heap);
    if (context->render_target)
        ID3D12Resource_Release(context->render_target);

    ID3D12CommandAllocator_Release(context->allocator);
    ID3D12CommandQueue_Release(context->queue);
    if (context->list1)
        ID3D12GraphicsCommandList1_Release(context->list1);
    ID3D12GraphicsCommandList_Release(context->list);

    if (context->device2)
        ID3D12Device2_Release(context->device2);

    refcount = ID3D12Device_Release(context->device);
    ok_(file, line)(!refcount, "ID3D12Device has %u references left.\n", (unsigned int)refcount);
}

static inline D3D12_CPU_DESCRIPTOR_HANDLE get_cpu_handle(ID3D12Device *device,
        ID3D12DescriptorHeap *heap, D3D12_DESCRIPTOR_HEAP_TYPE heap_type, unsigned int offset)
{
    D3D12_CPU_DESCRIPTOR_HANDLE handle;
    unsigned int descriptor_size;

    handle = ID3D12DescriptorHeap_GetCPUDescriptorHandleForHeapStart(heap);
    if (!offset)
        return handle;

    descriptor_size = ID3D12Device_GetDescriptorHandleIncrementSize(device, heap_type);
    handle.ptr += offset * descriptor_size;
    return handle;
}

static inline D3D12_GPU_DESCRIPTOR_HANDLE get_gpu_handle(ID3D12Device *device,
        ID3D12DescriptorHeap *heap, D3D12_DESCRIPTOR_HEAP_TYPE heap_type, unsigned int offset)
{
    D3D12_GPU_DESCRIPTOR_HANDLE handle;
    unsigned int descriptor_size;

    handle = ID3D12DescriptorHeap_GetGPUDescriptorHandleForHeapStart(heap);
    if (!offset)
        return handle;

    descriptor_size = ID3D12Device_GetDescriptorHandleIncrementSize(device, heap_type);
    handle.ptr += offset * descriptor_size;
    return handle;
}

static inline D3D12_CPU_DESCRIPTOR_HANDLE get_cpu_descriptor_handle(struct test_context *context,
        ID3D12DescriptorHeap *heap, unsigned int offset)
{
    return get_cpu_handle(context->device, heap, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, offset);
}

static inline D3D12_CPU_DESCRIPTOR_HANDLE get_cpu_sampler_handle(struct test_context *context,
        ID3D12DescriptorHeap *heap, unsigned int offset)
{
    return get_cpu_handle(context->device, heap, D3D12_DESCRIPTOR_HEAP_TYPE_SAMPLER, offset);
}

static inline D3D12_CPU_DESCRIPTOR_HANDLE get_cpu_rtv_handle(struct test_context *context,
        ID3D12DescriptorHeap *heap, unsigned int offset)
{
    return get_cpu_handle(context->device, heap, D3D12_DESCRIPTOR_HEAP_TYPE_RTV, offset);
}

static inline D3D12_CPU_DESCRIPTOR_HANDLE get_cpu_dsv_handle(struct test_context *context,
        ID3D12DescriptorHeap *heap, unsigned int offset)
{
    return get_cpu_handle(context->device, heap, D3D12_DESCRIPTOR_HEAP_TYPE_DSV, offset);
}

static inline D3D12_GPU_DESCRIPTOR_HANDLE get_gpu_descriptor_handle(struct test_context *context,
        ID3D12DescriptorHeap *heap, unsigned int offset)
{
    return get_gpu_handle(context->device, heap, D3D12_DESCRIPTOR_HEAP_TYPE_CBV_SRV_UAV, offset);
}

static inline D3D12_GPU_DESCRIPTOR_HANDLE get_gpu_sampler_handle(struct test_context *context,
        ID3D12DescriptorHeap *heap, unsigned int offset)
{
    return get_gpu_handle(context->device, heap, D3D12_DESCRIPTOR_HEAP_TYPE_SAMPLER, offset);
}

#endif  /* __VKD3D_D3D12_TEST_UTILS_H */
